/**
 * Expired Domains Checker - Content Script
 * Aggiunge colonne ETV e GMB alla tabella di expireddomains.net
 */

(async function() {
    'use strict';

    // Configurazione default
    const DEFAULT_CONFIG = {
        serverUrl: 'http://localhost:3847',
        geo: 'it',
        autoLoad: false,
        batchSize: 10
    };

    // Carica configurazione
    async function getConfig() {
        return new Promise((resolve) => {
            chrome.storage.sync.get(DEFAULT_CONFIG, (config) => {
                resolve(config);
            });
        });
    }

    // Trova la tabella dei domini
    function findDomainsTable() {
        // expireddomains.net usa una tabella con classe "base1"
        const table = document.querySelector('table.base1');
        if (!table) return null;
        return table;
    }

    // Estrae i domini dalla tabella
    function extractDomains(table) {
        const domains = [];
        const rows = table.querySelectorAll('tbody tr');

        rows.forEach((row, index) => {
            // La prima colonna contiene il checkbox, la seconda il dominio
            const domainCell = row.querySelector('td.field_domain a');
            if (domainCell) {
                const domain = domainCell.textContent.trim();
                domains.push({ domain, row, index });
            }
        });

        return domains;
    }

    // Aggiunge le colonne header
    function addHeaderColumns(table) {
        const headerRow = table.querySelector('thead tr');
        if (!headerRow) return false;

        // Verifica se le colonne esistono già
        if (headerRow.querySelector('.edc-etv-header')) return true;

        // Crea header ETV
        const etvHeader = document.createElement('th');
        etvHeader.className = 'edc-etv-header';
        etvHeader.innerHTML = '<span title="Estimated Traffic Value (DataForSEO)">ETV</span>';

        // Crea header GMB
        const gmbHeader = document.createElement('th');
        gmbHeader.className = 'edc-gmb-header';
        gmbHeader.innerHTML = '<span title="Google My Business">GMB</span>';

        // Inserisci dopo la colonna dominio
        const domainHeader = headerRow.querySelector('th.field_domain');
        if (domainHeader) {
            domainHeader.after(etvHeader, gmbHeader);
        } else {
            headerRow.appendChild(etvHeader);
            headerRow.appendChild(gmbHeader);
        }

        return true;
    }

    // Aggiunge le celle per ogni riga
    function addDataCells(domains) {
        domains.forEach(({ row }) => {
            // Verifica se le celle esistono già
            if (row.querySelector('.edc-etv-cell')) return;

            const etvCell = document.createElement('td');
            etvCell.className = 'edc-etv-cell';
            etvCell.innerHTML = '<span class="edc-loading">-</span>';

            const gmbCell = document.createElement('td');
            gmbCell.className = 'edc-gmb-cell';
            gmbCell.innerHTML = '<span class="edc-loading">-</span>';

            // Inserisci dopo la colonna dominio
            const domainCell = row.querySelector('td.field_domain');
            if (domainCell) {
                domainCell.after(etvCell, gmbCell);
            } else {
                row.appendChild(etvCell);
                row.appendChild(gmbCell);
            }
        });
    }

    // Chiama API ETV
    async function fetchETV(domains, serverUrl) {
        try {
            const response = await fetch(`${serverUrl}/api/etv`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ domains: domains.map(d => d.domain) })
            });

            if (!response.ok) throw new Error(`HTTP ${response.status}`);

            const data = await response.json();
            return data.success ? data.results : {};
        } catch (error) {
            console.error('[EDC] ETV Error:', error);
            return {};
        }
    }

    // Chiama API GMB
    async function fetchGMB(domains, serverUrl, geo) {
        try {
            const response = await fetch(`${serverUrl}/api/gmb-claude`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    domains: domains.map(d => d.domain),
                    geo: geo
                })
            });

            if (!response.ok) throw new Error(`HTTP ${response.status}`);

            const data = await response.json();
            return data.success ? data.results : {};
        } catch (error) {
            console.error('[EDC] GMB Error:', error);
            return {};
        }
    }

    // Formatta valore ETV
    function formatETV(value) {
        if (value === undefined || value === null) return '-';
        if (value >= 1000000) return (value / 1000000).toFixed(1) + 'M';
        if (value >= 1000) return (value / 1000).toFixed(1) + 'K';
        return value.toString();
    }

    // Aggiorna celle con risultati
    function updateCells(domains, etvResults, gmbResults) {
        domains.forEach(({ domain, row }) => {
            const etvCell = row.querySelector('.edc-etv-cell');
            const gmbCell = row.querySelector('.edc-gmb-cell');

            if (etvCell) {
                const etvData = etvResults[domain];
                if (etvData && etvData.etv !== undefined) {
                    const etv = etvData.etv;
                    etvCell.innerHTML = `<span class="edc-value ${etv > 10000 ? 'edc-high' : etv > 1000 ? 'edc-medium' : ''}" title="${etv.toLocaleString()}">${formatETV(etv)}</span>`;
                } else if (etvData && etvData.error) {
                    etvCell.innerHTML = '<span class="edc-error" title="Errore">!</span>';
                } else {
                    etvCell.innerHTML = '<span class="edc-none">0</span>';
                }
            }

            if (gmbCell) {
                const gmbData = gmbResults[domain];
                if (gmbData && gmbData.hasGMB === true) {
                    gmbCell.innerHTML = '<span class="edc-gmb-yes" title="Ha scheda GMB">SI</span>';
                } else if (gmbData && gmbData.hasGMB === false) {
                    gmbCell.innerHTML = '<span class="edc-gmb-no" title="Nessuna scheda GMB">NO</span>';
                } else if (gmbData && gmbData.error) {
                    gmbCell.innerHTML = '<span class="edc-error" title="Errore">!</span>';
                } else {
                    gmbCell.innerHTML = '<span class="edc-loading">-</span>';
                }
            }
        });
    }

    // Processa domini in batch
    async function processBatch(domains, config, startIndex, batchSize) {
        const batch = domains.slice(startIndex, startIndex + batchSize);
        if (batch.length === 0) return;

        console.log(`[EDC] Processing batch ${startIndex}-${startIndex + batch.length} of ${domains.length}`);

        // Segna come loading
        batch.forEach(({ row }) => {
            const etvCell = row.querySelector('.edc-etv-cell');
            const gmbCell = row.querySelector('.edc-gmb-cell');
            if (etvCell) etvCell.innerHTML = '<span class="edc-loading">...</span>';
            if (gmbCell) gmbCell.innerHTML = '<span class="edc-loading">...</span>';
        });

        // Fetch parallelo ETV e GMB
        const [etvResults, gmbResults] = await Promise.all([
            fetchETV(batch, config.serverUrl),
            fetchGMB(batch, config.serverUrl, config.geo)
        ]);

        // Aggiorna celle
        updateCells(batch, etvResults, gmbResults);
    }

    // Aggiunge pulsante di controllo
    function addControlButton(table, domains, config) {
        // Cerca container esistente o crea nuovo
        let container = document.querySelector('.edc-controls');
        if (!container) {
            container = document.createElement('div');
            container.className = 'edc-controls';
            table.parentNode.insertBefore(container, table);
        }

        container.innerHTML = `
            <div class="edc-control-bar">
                <button id="edc-load-btn" class="edc-btn edc-btn-primary">
                    Carica ETV + GMB (${domains.length} domini)
                </button>
                <button id="edc-load-selected-btn" class="edc-btn">
                    Solo selezionati
                </button>
                <span class="edc-status" id="edc-status"></span>
                <span class="edc-config">
                    Server: <code>${config.serverUrl}</code> | Geo: <code>${config.geo.toUpperCase()}</code>
                </span>
            </div>
        `;

        // Event listener per carica tutti
        document.getElementById('edc-load-btn').addEventListener('click', async () => {
            const btn = document.getElementById('edc-load-btn');
            const status = document.getElementById('edc-status');

            btn.disabled = true;
            btn.textContent = 'Caricamento...';

            for (let i = 0; i < domains.length; i += config.batchSize) {
                status.textContent = `Elaborazione ${i + 1}-${Math.min(i + config.batchSize, domains.length)} di ${domains.length}...`;
                await processBatch(domains, config, i, config.batchSize);

                // Piccola pausa tra batch per non sovraccaricare
                if (i + config.batchSize < domains.length) {
                    await new Promise(r => setTimeout(r, 500));
                }
            }

            status.textContent = 'Completato!';
            btn.textContent = `Ricarica ETV + GMB (${domains.length} domini)`;
            btn.disabled = false;
        });

        // Event listener per carica selezionati
        document.getElementById('edc-load-selected-btn').addEventListener('click', async () => {
            const selectedDomains = domains.filter(({ row }) => {
                const checkbox = row.querySelector('input[type="checkbox"]');
                return checkbox && checkbox.checked;
            });

            if (selectedDomains.length === 0) {
                alert('Seleziona almeno un dominio');
                return;
            }

            const btn = document.getElementById('edc-load-selected-btn');
            const status = document.getElementById('edc-status');

            btn.disabled = true;
            status.textContent = `Elaborazione ${selectedDomains.length} domini selezionati...`;

            for (let i = 0; i < selectedDomains.length; i += config.batchSize) {
                await processBatch(selectedDomains, config, i, config.batchSize);
                if (i + config.batchSize < selectedDomains.length) {
                    await new Promise(r => setTimeout(r, 500));
                }
            }

            status.textContent = 'Completato!';
            btn.disabled = false;
        });
    }

    // Inizializzazione
    async function init() {
        console.log('[EDC] Expired Domains Checker inizializzato');

        const config = await getConfig();
        console.log('[EDC] Config:', config);

        const table = findDomainsTable();
        if (!table) {
            console.log('[EDC] Tabella domini non trovata');
            return;
        }

        const domains = extractDomains(table);
        if (domains.length === 0) {
            console.log('[EDC] Nessun dominio trovato');
            return;
        }

        console.log(`[EDC] Trovati ${domains.length} domini`);

        // Aggiungi colonne
        addHeaderColumns(table);
        addDataCells(domains);

        // Aggiungi controlli
        addControlButton(table, domains, config);

        // Auto-load se abilitato
        if (config.autoLoad) {
            document.getElementById('edc-load-btn').click();
        }
    }

    // Avvia quando DOM è pronto
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

})();
